#! /usr/bin/env python
#############################################################################
#
#              GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/Classes.py,v $
# Version:     $RCSfile: Classes.py,v $ $Revision: 1.8 $
# Copyright:   (C) 1997-1999, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
Classes

The bytecode interpreter operates on instances of a set of Python
classes representing the basic datatypes of NewtonScript.  These
classes are implemented in Python, with equivalent semantics to the
NewtonScript implementation, but no represenational similarity.

Each class has the ability to marshall to and from a byte stream
representations of itself that match those used by the Newton,
allowing reading and writing Newton-format packages, for example.


Note that the views of the NewtonScript data types presented by the
Apple Formats document, and the NewtonScript Language document are
slightly different:

Language:			Formats:

all objects			refs
-- primitive classes		-- immediate
   -- immediate			   -- integer
      -- integer		   -- char
      -- char			   -- special
      -- boolean		      -- true
   -- binary			      -- nil
      -- symbol			      -- kSymbolClass
      -- string			      -- others ... ?
      -- real			-- pointer
   -- array			   -- pointer
   -- frame			   -- magic pointer
				objects
				-- binary
				-- array
				-- frame

The overlap here is fairly obvious, and the view taken by the language
book is that of a NewtonScript programmer.  This module, however, is
designed to support the virtual machine, and so must take a bias
towards the categorisations of the Formats doc.

It seems also that NOS2 extended these sets, with the addition of
various string classes, and perhaps the weak arrays seen with
ViewFrame.  Additionally, the various function classes (newFunc,
newCFunc) should be handled here?


"""

#############################################################################

import struct

from   GnutOS.Constants      import kSymbolClass
from   GnutOS.Exceptions     import BadSymbolString, GnutOSException


#############################################################################

REF_INTEGER  = "Integer"
REF_CHAR     = "Char"
REF_POINTER  = "Pointer"
REF_MAGIC    = "Magic"
REF_SPECIAL  = "Special"

OBJ_BINARY   = 0
OBJ_ARRAY    = 1
OBJ_FRAME    = 3


SYMVAL_STRING = "string"
SYMVAL_REAL = "real"


#############################################################################

class MarshallingException(GnutOSException):
    """Base exceptions class for all marshalling errors."""
    pass

class UnMarshallingConsistencyException(MarshallingException):
    """A consistency check failed while unmarshalling."""
    pass

class UnrecognisedReferenceTypeException(MarshallingException):
    """A Reference had an unrecognised type code (low two bits)."""
    pass


#############################################################################

class GnutData:
    """Base class for all NewtonScript data items."""

    def __init__(self):
	self._primitive = None
	self._class = None

    def PrimClassOf(self):
	return self._primitive

    def ClassOf(self):
	return self._class

    def IsSubclass(self, klass):
	pass

    def IsInstance(self, klass):
	pass

    
    def isNil(self):
        return 0

    def isTrue(self):
        return 0
    
    def __repr__(self):
        return "<GnutData Instance>"
    

class Ref(GnutData):
    """Base class for primitive objects in NewtonScript."""

    def __init__(self):
	self._primitive = None
	self._type = None
	self._value = None
	return


    def type(self):
	return self._type


    def pack(self, s, o):
	return o


    def unpack(self, s, o):
	"""Unpack a reference."""

	ref_fmt = ">l"                      # gotta be signed to use 32bit int type
	ref_len = struct.calcsize(ref_fmt)

	ref = struct.unpack(ref_fmt, s[o:o+ref_len])
	self._value = int(ref[0])

	if self._value & 0x3 == 0x3:
	    self._type = REF_MAGIC
	elif self._value & 0x3 == 0x1:
	    self._type = REF_POINTER
	elif self._value & 0x3 == 0x0:
	    self._type = REF_INTEGER
	elif self._value & 0xfff0000f == 0xa:
	    self._type = REF_CHAR
	elif self._value & 0x3 == 0x2:
	    self._type = REF_SPECIAL
	else:
	    raise UnrecognisedReferenceTypeException(hex(self._value))

        #print self._type, hex(self._value)
	return o + ref_len


    def __repr__(self):
        return "<Ref instance>"

    
#############################################################################
#  ref sub-classes

class ImmediateRef(Ref):
    def __init__(self):
	self._primitive = "immediate ref"

    def value(self):
        return self._value


class PointerRef(Ref):
    def __init__(self):
	self._primitive = "pointer ref"


#############################################################################
#  immediate sub-classes

class Integer(ImmediateRef):
    def __init__(self, i=None):
	ImmediateRef.__init__(self)
	self._type = REF_INTEGER
	self._value = i or 0


    def unpack(self, s, o):
	"""Set the value of an Integer from the stream supplied."""

	int_fmt = ">l"
	int_len = struct.calcsize(int_fmt)

	v = struct.unpack(int_fmt, s[o:o+int_len])
	self._value = v[0] >> 2

	return o + int_len


    def pack(self, s, o):
	"""Write the value of Integer to the supplied stream."""

	#-- pack int into 32, with low 2 bits == 00
	str_v = struct.pack("L", (self._value << 2))
	len_v = struct.calcsize("L")

	#-- write to stream
	s[o:o+len_v] = str_v

	#-- return new offset
	return o+len_v


    def __repr__(self):
        return "<Integer %d (%s)>" % (self._value, hex(self._value))
    

class Char(ImmediateRef):
    def __init__(self):
	ImmediateRef.__init__(self)
	self._type = REF_CHAR
	self._value = ""

    def pack(self, s, o):
	"""Write the value of Char to the supplied stream."""

	pass

    def unpack(self, s, o):
	char_fmt = ">L"
	char_len = struct.calcsize(char_fmt)

	v = struct.unpack(char_fmt, s[o:o+char_len])
	self._value = int(v[0]) & 0x000ffff0  # unicode mask

	return o + char_len
	
    def __repr__(self):
        return "<Char '%s' (%s)>" % (self._value & 0xff, hex(self._value))
    

class Special(ImmediateRef):
    def __init__(self, value=0):
	ImmediateRef.__init__(self)
	self._type = REF_SPECIAL
	self._value = value
	return

    def unpack(self, s, o):
	spec_fmt = ">L"
	spec_len = struct.calcsize(spec_fmt)

	v = struct.unpack(spec_fmt, s[o:o+spec_len])
        self._value = int(v[0])

	return o + spec_len

    def isNil(self):
        return (self._value == 0x2)

    def isTrue(self):
        return (self._value == 0x1a)
    
    def __repr__(self):
        if self._value == 0x2:
            return "NIL"
        elif self._value == 0x1a:
            return "TRUE"
        elif self._value == 0x55552:
            return "kSymbolClass"
        else:
            return "<Special [%s]>" % (str(hex(self._value)),)

    
#############################################################################
#  pointer sub-classes

class Pointer(PointerRef):
    def __init__(self):
	PointerRef.__init__(self)
	self._type = REF_POINTER
	self._value = 0            # offset in package of target
        self._target = None        # target object
        

    def unpack(self, pkg, s, o):
	ptr_fmt = ">L"
	ptr_len = struct.calcsize(ptr_fmt)

	v = struct.unpack(ptr_fmt, s[o:o+ptr_len])
	self._value = int(v[0] - 1)  # remove the lower two bits (0x1)

	if pkg._d_ptr.has_key(self._value):
            self._target = pkg._d_ptr[self._value]
            #print "pointer alias for target offset %d (%s)" % (self._value, self._target)
            
        else:
            #print "unpacking pointer@%d: offset=%d" % (o, self._value)
            obj = Object()
            obj.unpack_header(pkg, s, self._value)

            flags = obj.type()

            if flags & 3 == OBJ_ARRAY:
                #print " --> target is array"
                obj = Array()
                obj.unpack(pkg, s, self._value)

            elif flags & 3 == OBJ_FRAME:
                #print " --> target is frame"
                obj = Frame()
                obj.unpack(pkg, s, self._value)

            elif flags & 3 == OBJ_BINARY:
                #print " --> target is binary"
                obj = Binary()
                obj.unpack(pkg, s, self._value)

                #-- get binary's class ref
                classref = obj.getClass()

                if classref.type() == REF_SPECIAL and classref.value() == kSymbolClass:
                    obj = Symbol()
                    obj.unpack(pkg, s, self._value)

                elif classref.type() == REF_POINTER:

                    classobj = classref.target()
                    if classobj.type() == OBJ_BINARY and \
                       classobj.getClass() == kSymbolClass:

                        if classobj.name() == SYMVAL_STRING:
                            obj = String()
                            obj.unpack(pkg, s, self._value)

                        elif classobj.name() == SYMVAL_REAL:
                            obj = Real()
                            obj.unpack(pkg, s, self._value)

                        else:
                            #print "    --> binary is", classobj.name()
                            pass
                            
                else:
                    print "    ### classref type =", classref.type()
                    
            else:
                print "### unknown object type", hex(flags)

            self._target = obj
            pkg._d_ptr[self._value] = obj
            
	return o + ptr_len


    def pack(self):
	return struct.pack(">L", ((self._value & 0xffffff00) + 1))
    
    def target(self):
        return self._target
    
    def index(self):
	return self._value

    def __repr__(self):
        return "<Pointer @%d = %s>" % (self._value, str(self._target))
    

class Magic(PointerRef):
    def __init__(self, table=0, index=0):
	PointerRef.__init__(self)
	self._type = REF_MAGIC
	self._table = table
	self._index = index

    def unpack(self, s, o):
	magic_fmt = ">H H"
	magic_len = struct.calcsize(magic_fmt)

	table, idx = struct.unpack(magic_fmt, s[o:o+magic_len])

	self._table = table
	self._index = idx >> 2

	return o + magic_len

    def  __repr__(self):
        return "<Magic %d@%d>" % (self._table, self._index)

    
#############################################################################
#  object & sub-classes

class Object(GnutData):
    """Base class for classed objects in NewtonScript."""

    def __init__(self):
	self._primitive = None
	self._flags = 0
	self._size = 0
	return


    def unpack_header(self, pkg, s, o):
	hdr = s[o:o+4]
	self._size = hdr[:3]
	self._flags = ord(hdr[3])
        return


    def type(self):
        return self._flags


    def typeName(self):
        return ["Binary", "Array", "", "Frame"][self._flags]

    
class Binary(Object):
    def __init__(self, classRef=None, data=None):
	"""Create a new Binary object."""

	Object.__init__(self)
	#self._primitive = SYM_BINARY  #fixme: chicken&egg with symbol

	self._class = classRef
	self._data = data
	return


    def unpack(self, pkg, s, o):
	bin_fmt = ">l l"
	bin_len = struct.calcsize(bin_fmt)
	flgs, junk = struct.unpack(bin_fmt, s[o:o+bin_len])

	#-- unpack class reference
	c = Ref()
	c.unpack(s, o+bin_len)

	if c.type() == REF_SPECIAL:
            c = Special()
            c.unpack(s, o+bin_len)
	elif c.type() == REF_POINTER:
            c = Pointer()
            c.unpack(pkg, s, o+bin_len)
        elif  c.type() == REF_CHAR:
            c = Char()
            c.unpack(s, o+bin_len)
        elif c.type() == REF_INTEGER:
            c = Integer()
            c.unpack(s, o+bin_len)
        elif  c.type() == REF_MAGIC:
            c = Magic()
            c.unpack(s, o+bin_len)
	else:
            raise "BogusBinaryClassRef", c.type()

        self._class = c
        
	#-- unpack data
	size = int(flgs >> 8)
	self._data = s[o+12: o+size]    # 12 = 3 x 32 bits in obj header & classref

	return o + size


    def setClass(self, classSym):
	"""Set the class of this binary object."""

	#fixme: check classSym is a symbol?
	self._class = classSym
	return


    def getClass(self):
	return self._class


    def getData(self):
	return self._data


    def __repr__(self):
        return "<Binary (%s), length=%d>" % (self._class, len(self._data))
    

class Array(Object):
    """
    """

    def __init__(self, lst_init = None):
	Object.__init__(self)

	self._primitive = SYM_ARRAY
	self._len = 0       # number of slots
	self._items = {}    # use a dictionary to allow sparse indexing

	#-- initialise from Python list if provided
	if lst_init:
	    cnt = 0
	    for item in lst_init:
		self._items[cnt] = item
		cnt = cnt + 1

	    self._len = len(self._items)

	return


    def pack(self, s, o):
	pass


    def unpack(self, pkg, s, o):
	"""Unpack an array object from the supplied string."""

	arr_fmt = ">L L L"
	arr_len = struct.calcsize(arr_fmt)

	flgs, junk, klass = struct.unpack(arr_fmt, s[o:o+arr_len])

	if int(flgs) & 0xff != 0x41:
	    raise UnMarshallingConsistencyException("Array flags = %x" % flgs)

	self._size  = int(flgs >> 8)
	#print "unpacking array@%d, size=%d" % (o, self._size)
        
	self._len   = int((self._size - (4 * 3)) / 4)
	self._class = klass

	o = o + arr_len

	for idx in range(self._len):
	    i = Ref()
	    i.unpack(s, o)
	    reftype = i.type()

	    if reftype == REF_INTEGER:
		item = Integer()
		o = item.unpack(s, o)

	    elif reftype == REF_POINTER:
		item = Pointer()
		o = item.unpack(pkg, s, o)

	    elif reftype == REF_CHAR:
		item = Char()
		o = item.unpack(s, o)

	    elif reftype == REF_SPECIAL:
		item = Special()
		o = item.unpack(s, o)

	    elif reftype == REF_MAGIC:
		item = Magic()
		o = item.unpack(s, o)

	    else:
		print "fuck! what's this?", reftype
		raise UnrecognisedReferenceTypeException(reftype)

	    self._items[idx] = item

	return o


    def __getitem__(self, n):
	"""Return the nth item from the array."""

	return self._items[n]


    def __setitem__(self, n, i):
	"""Set the nth item to i."""

	self._items[n] = i
	return


    def __delitem__(self, n):
	"""Remove the nth item from the array."""

	self._items[n] = None
	return


    def __len__(self):
	return len(self._items)

    def __repr__(self):
        return "[Array (%d) %s]" % \
               (len(self._items),
                len(self._items) and reduce(lambda s,i,a=self._items: s+", "+str(a[i]),
                                            range(1, len(self._items)),
                                            str(self._items[0])) or "")

    
class Frame(Object):
    """newtonscript frames are LISP-like, slotted collections"""

    def __init__(self, *vargs, **kargs):
	Object.__init__(self)

	self._primitive = SYM_FRAME
	self._items = {}
	self._len = 0

        self._slots = []
        self._map = {}

	for key in kargs.keys():    #fixme: update() (available only from 1.5.2?)
            self._slots.append(kargs[key])
            self._map[key] = len(self._slots) - 1

	return

    def __getitem__(self, name):
	return self._slots[self._map[name]]

    def __setitem__(self, name, value):
        if self._map.has_key(name):
            self._slots[self._map[name]] = value

        else:
            self._slots.append(value)
            self._map[name] = len(self._slots) - 1
            
    def __len__(self):
	return len(self._slots)

    def __delitem__(self, name):
	del self._slots[self._map[name]]
        del self._map[name]

    def has_key(self, name):
	return self._map.has_key(name)

    def keys(self):
	return self._map.keys()

    def nth(self, n):
	return self._slots[n]

    def unpack(self, pkg, s, o):
	"""Unpack a frame from the supplied string."""

	frm_fmt = ">L L"
	frm_len = struct.calcsize(frm_fmt)

	flgs, junk = struct.unpack(frm_fmt, s[o:o+frm_len])

	if flgs & 0xff != 0x43:
	    raise "FuckOff!", "unpacking error"

	self._size = flgs >> 8
	self._len = (self._size - (3 * 4)) / 4

	o = o + frm_len

	#-- unpack frame map pointer
	map_ptr = Pointer()
	o = map_ptr.unpack(pkg, s, o)

	#-- unpack slots
	tmp_items = {}
	for idx in range(self._len):
	    i = Ref()
	    i.unpack(s, o)
	    reftype = i.type()

	    if reftype == REF_INTEGER:
		item = Integer()
		o = item.unpack(s, o)

	    elif reftype == REF_POINTER:
		item = Pointer()
		o = item.unpack(pkg, s, o)

	    elif reftype == REF_CHAR:
		item = Char()
		o = item.unpack(s, o)

	    elif reftype == REF_SPECIAL:
		item = Special()
		o = item.unpack(s, o)

	    elif reftype == REF_MAGIC:
		item = Magic()
		o = item.unpack(s, o)

	    else:
		print "fuck! what's this?", reftype
		raise "FUCK", "bad ref"

	    self._slots.append(item)

	#-- unpack the map
        d_map = self._unpack_map(map_ptr.target())
        for idx in range(self._len):
            self._map[d_map[idx]] = idx

	return o


    def _unpack_map(self, frm_map):
        """(Internal). Recursively unpack the frame map array."""

        d_map = {}
        
        supermap_ptr = frm_map[0]
        if supermap_ptr.type() != REF_SPECIAL or not supermap_ptr.isNil():
            print "recursing to supermap"
            d_map.update(self._unpack_map(supermap_ptr.target()))

        offset = len(d_map)
        for idx in range(len(frm_map) - 1):
            d_map[idx + offset] = frm_map[idx + 1].target().name()
            
        return d_map

    def __repr__(self):
        keys = self._map.keys()
        return "{Frame (%d) %s}" % \
               (self._len,
		self._len and \
                reduce(lambda o,n,m=self._map,s=self._slots: o+", %s:%s" % (n,str(s[m[n]])),
                       keys[1:],
                       "%s:%s" % (keys[0], str(self._slots[self._map[keys[0]]]))) or "")
                

    
#############################################################################
#  binary subclasses

class Symbol(Binary):
    """

    Symbols are one of the very basic classes of objects in
    NewtonScript.  They are used as names for pretty much everything.
    This is because they have a quick hash value which makes lookup
    fast.

    """

    def __init__(self, name=None):
	"""Create a new symbol."""

	Binary.__init__(self)
	self._class = kSymbolClass    # special ref
	self._hash = 0

	if name:
	    self.setName(name)
	else:
	    self._name = name

	return

    def unpack(self, pkg, s, o):
	sym_fmt = ">L L L L"
	sym_len = struct.calcsize(sym_fmt)

	flgs, junk, klass, hash = struct.unpack(sym_fmt, s[o:o+sym_len])

	size = int(flgs >> 8)
	name_o = o + sym_len

        self._name = ""
	while s[name_o] != "\0":
	    self._name = self._name + s[name_o]
	    name_o = name_o + 1
	
        self.__set_hash(self._name)

        #print "unpacked %s" % str(self)
	return o + size


    def name(self):
	"""Return the symbol's string value."""
	return self._name


    def hash(self):
	"""Return the symbol's hash value."""
	return self._hash


    def setName(self, name):
	"""Set the string name for the symbol.

	*name*     -- string value to set as symbol contents
	Returns    -- nothing
	Exceptions -- none

	The algorithm to calculate the hash value is specified on page
        1-15 of Apple's Newton Formats document, version 1.1 """

	if hasattr(self, "_name") and self._name:
	    raise SymbolNameIsImmutable(name, self._name)

	self._name = name
        self.__set_hash(name)
        return


    def __set_hash(self, name):
	#-- calculate the hash value
	res = 0L
	lc_offset = ord("a") - ord("A")

	for c in name:
	    if ord(c) < 32 or ord(c) > 127:
		raise BadSymbolString(name, c)

	    if c >= "a" and c <= "z":
		res = res + ord(c) - lc_offset
	    else:
		res = res + ord(c)

	self._hash = (res * 2654435769L) & 0xffffffffL  # 3 force to 32 bit

	return


    def __hash__(self):
	"""Returns value to be used in Python dictionary indexes."""
	return hash(self._name)

    def __cmp__(self, other):
	if hash(self) < hash(other):
	    return -1
	elif hash(self) == hash(other):
	    return 0
	else:
	    return 1

    def __repr__(self):
	return "Symbol (%s) |%s|" % (hex(self._hash), self._name)


class String(Binary):
    """NewtonScript String class."""

    def __init__(self, str_init = None):
	"""Create new String object."""

	Binary.__init__(self)
	self._class = SYM_STRING    #fixme: pointer to sym_string?

	if not str_init:
	    self._value = ""
	else:
	    self._value = str_init

	return


    def unpack(self, pkg, s, o):
        ret = Binary.unpack(self, pkg, s, o)

        self._str = ""
	f = 0
	for b in self._data:
	    if not f:
		f = 1
	    else:
		f = 0
		if ord(b) != 0:
		    self._str = self._str + b

	#print "unpacked %s" % str(self)
	return ret

        
    def python(self):
	return self._str

#    def __str__(self):
#	return self._str

    def __len__(self):
	return len(self._str)

    def __repr__(self):
        return 'String "%s"' % (self._str,)

    
class Real(Binary):
    def __init__(self):
	Binary.__init__(self)
	self._class = SYM_REAL
	self._value = 0.0

    def python(self):
	return self._value

    def __repr__(self):
        return "Float X"
    

#############################################################################
    
SYM_IMMEDIATE = Symbol("Immediate")
SYM_BINARY    = Symbol("Binary")
SYM_FRAME     = Symbol("Frame")
SYM_ARRAY     = Symbol("Array")
SYM_BOOLEAN   = Symbol("Boolean")
SYM_STRING    = Symbol("string")
SYM_REAL      = Symbol("real")

#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
